/*=============================================================================
 
FFTtest
=============   ====================================
 
Dateiname:      main.cpp 
 
Autors:		    Tim Fischer
 
Version:        0.1 of 03.08.2022
 
Hardware:       SimulIDE 1.0.1 R1290
				NOT TESTED ON REAL HARDWARE!
				ONe has to check whether the ADC frequency is still feasible on real hardware.
 
Software:       Dev Environment:	AtmelStudio 7.0
                C-Compiler:			AVR/GNU C Compiler 5.4.0
 
Function:       Shows max FFT sample and 16 bins with 
 
Display: 
                +----------------+   
                |           = 5  |   
                |0000001910000000|   
                +----------------+   
  
=============================================================================*/ 
 
// Declarations ==============================================================
 
// Quarz frequency
#ifndef F_CPU							    // If not defined
#define F_CPU 12288000UL					// Set to 12.288 MHz
#endif                          
 
// Include of Header-Files
#include <avr/io.h>							// I/O Configuration 
#include <util/delay.h>						// Definition of Delays (Wait timers)
#include <stdbool.h>						// library for Bit variables
#include <string.h>							// library for String variables
#include "lcd_lib_de.h"						// library for LCD-Display
#include "arduinoFFT.h"						// Fast Fourier Transformation, initially for Arduino, but works for AVR w/o Arduino, too
   
// Makros
#define SET_BIT(BYTE, BIT)  ((BYTE) |=  (1 << (BIT))) // Set   bit in byte 
#define CLR_BIT(BYTE, BIT)  ((BYTE) &= ~(1 << (BIT))) // clear bit in byte 

#define LCD_LENGTH 16						// Number of characters in a single LDC line

arduinoFFT FFT = arduinoFFT();				// Create FFT object 

const uint16_t samples = 128;				// Sample number of FFT; this value MUST ALWAYS be a power of 2
const uint16_t samplesInOneBin = samples>>5;// Number of Samples for Bin
uint16_t strongestSample = 0;				// Variable for Sample with highest value 
uint16_t bin[LCD_LENGTH];					// Bins for combining FFT samples
uint16_t binMaxNr=0;						// Bin number with the highest value

const double samplingFrequency = 26000;		// Sampling frequency, given by F_CPU and ADC freq

double vReal[samples];						// Input AND output arrays; input arrays receive computed results from FFT 
double vImag[samples];						// They initially shall contain the measured values

uint8_t adcSamples[samples];				// Array for sampled ADC values
uint8_t adcFinished=0;						// Flag, showing whether the sampling of 128 values is finished

// Function prototypes
void adc_init(void);						// Initializing the ADC

// Main program ==============================================================
int main()                    
{   
 	lcd_init();								// Initializing the LCD
    adc_init();								// Initializing the ADC
     
	char outputString[16];					// Contains the string to be printed on the LCD
 
    while(1)								// infinite loop
    {
		if  (adcFinished)					// only do something, when 128 ADC values are sampled
		{
			adcFinished = 0;													// reset flag
			for (uint16_t sampleNr = 0; sampleNr < samples; sampleNr++)			// For the whole FFT input array
			{
				vReal[sampleNr] = float(adcSamples[sampleNr]);						// Set real part to ADC values
				vImag[sampleNr] = 0.0;												// Imaginary part must be zeroed in case of looping to avoid wrong calculations and overflows
			};
			FFT.DCRemoval();													// Get rid of DC component
			FFT.Windowing(vReal, samples, FFT_WIN_TYP_HAMMING, FFT_FORWARD);	// Weigh data 
			FFT.Compute(vReal, vImag, samples, FFT_FORWARD);					// Compute FFT 
			FFT.ComplexToMagnitude(vReal, vImag, samples);						// Compute magnitudes 
		
			vReal[0]=0.0; 	vReal[1]=0.0;										// Lowest frequencies have high amplitude, due to DC component and windowing
																				// These have to be neglected (best would be a low pass filter, easiest is setting to 0) 
			for(uint8_t sampleNr=1;sampleNr<(samples>>1);sampleNr++)			// Find FFT sample with the highest values
				if( vReal[sampleNr]> vReal[strongestSample]) strongestSample=sampleNr;

			sprintf(outputString, "= %i\r\n", strongestSample);					// Put the number strongestSample into string outputString 
			lcd_displayMessage(outputString, 0, 16-strlen(outputString));		// Show strongestSample

			binMaxNr=0;															// Reset binMaxNr = Nr of bin with the highest value
			for(uint8_t binNr=0 ; binNr<16 ; binNr++)							// Go through all bins (highest value is 16 due to the LCD display size)
			{
				bin[binNr]=0;														// Reset actual bin 
				for(uint8_t i=0;i<samplesInOneBin;i++)								// Go through the samplesInOneBin
					bin[binNr] += uint16_t( vReal[ binNr*4 + i ] );					// Add up the vReal samples into the actual bin
				if (bin[binNr] > bin[binMaxNr]) binMaxNr = binNr ;					// Check whether actual bin has the highest value compared to all bins before
			};
			
			for(uint8_t binNr=0;binNr<16;binNr++)								// Go again through all bins (now, the highest value in a bin is known)
				outputString[binNr]= '0' + uint8_t (float(bin[binNr])/bin[binMaxNr]*9); // For the output set the highest value to 9 and all other linearly smaller

			lcd_displayMessage(outputString, 1, 0);								// Print bins on display 
			_delay_ms(10);														// Wait 10 ms
		}
    }                          
}                           
 
// Functions =================================================================
 
// Initializing of ADC
void adc_init(void)
{
	ADMUX   |= (1<<REFS0) | (1<<ADLAR) ;// Vref =AVCC; ADC0
	ADCSRA  |= (1<<ADPS0)
			|  (1<<ADPS2)				// Prescaler 32 --> might cause problems in real applications 
			|  (1<<ADATE)				// Free Running on
			|  (1<<ADIE)				// Interrupt Enable on
			|  (1<<ADEN);				// ADC on
	ADCSRA |= (1<<ADSC);				// Start ADC
	(void) ADCH;						// Ignore first value
}

// ADC interrupt. Gets executed every time, when a new value was sampled.
ISR (ADC_vect)
{
	static uint8_t adcCount=0;			//
	adcSamples[adcCount] = ADCH;		// put ADC value first in uint8 variable --> faster!
	adcCount++;							// Increment counter
	if (adcCount<=samples) return;		// When counter smaller than 128 exit
	adcCount	= 0 ;						// Else: reset counter and rise adcFinished flag
	adcFinished	= 1 ;
}

