/* 
 * LCD driver PCD8544 
 * 
 * Adapted by Tim Fischer
 * Based on code of Marian Hrinko (mato.hrinko@gmail.com)
 *
 * Author			Tim Fischer, Marian Hrinko
 * Date				23.05.2020
 * Description		Library designed for LCD with PCD8544 driver
 *					LCD Resolution 48x84
 *					Communication through 5 control wires (SCK, RST, DIN, CE, CS)
 */

#include <stdio.h>
#include <string.h>
#include <avr/io.h>
#include <util/delay.h>
#include "pcd8544.h"
#include "characterset5x8.h"

#define PCD_WIDTH				84				// LCD is	84 pixels wide
#define PCD_HEIGHT				48				//     and	48 pixels high

#define PCD_CHAR_WIDTH			6				// a single character is	 5 pixels wide (incl. 1 pixel beween characters)
#define PCD_CHAR_HEIGHT			8				//     					and	 8 pixels high (incl. 1 pixel beween lines)


#define PCD_EXTENDEDINSTRUCTION 0x01			// Function set, Extended instruction set control
#define PCD_ENTRYMODE			0x02			// Function set, Entry mode
#define PCD_POWERDOWN			0x04			// Function set, Power down mode

#define PCD_DISPLAYBLANK		0x00			// Display control, blank
#define PCD_DISPLAYALLON		0x01			// Display control, all segments on
#define PCD_DISPLAYNORMAL		0x04			// Display control, normal mode
#define PCD_DISPLAYINVERTED		0x05			// Display control, inverse mode

#define PCD_DISPLAYCONTROL		0x08			// Basic instruction set - Set display configuration
#define PCD_FUNCTIONSET			0x20			// Basic instruction set
#define PCD_SETYADDR			0x40			// Basic instruction set - Set Y address of RAM, 0 <= Y <= 5
#define PCD_SETXADDR			0x80			// Basic instruction set - Set X address of RAM, 0 <= X <= 83

#define PCD_SETTEMP				0x04			// Extended instruction set - Set temperature coefficient
#define PCD_SETBIAS				0x10			// Extended instruction set - Set bias system
#define PCD_SETVOP				0x80			// Extended instruction set - Write voltage of operation to register

#define PCD_FIRST_ASC			0x20			// ASCII code of the first character
#define PCD_LAST_ASC			0x7F			// ASCII code of the last  character

#define CMD						0x00			// Flag for command type signal
#define DATA					0x01			// Flag for    data type signal

#define	SET_BIT(PORT, BIT)	((PORT) |=  (1 << (BIT))) // Set	bit on port
#define CLR_BIT(PORT, BIT)	((PORT) &= ~(1 << (BIT))) // Clear	bit on port 

static char pcd_displayBuffer[PCD_BUFFER_SIZE];	// Cache memory Lcd 6 * 84 = 504 bytes
int			pcd_BufferIndex = 0;				// Cache memory char index

// reset impulse to enable display
void pcd_enable(void)
{
	_delay_ms(1);								// delay 1ms
	CLR_BIT(PCD_PORT, PCD_RST_PIN);				// Reset Low
	_delay_ms(1);								// delay 1ms
	SET_BIT(PCD_PORT, PCD_RST_PIN);				// Reset High
}

// initialize the pcd8544 controller
void pcd_init(void)
{
	PCD_PORT	|=	(1 << PCD_RST_PIN);			// Activate pull-up register -> logical high on pin RST

	PCD_DDR		|=	(1 << PCD_RST_PIN)	|		// Set RST, SCK, DIN, CE, DC to output
					(1 << PCD_CLK_PIN)	| 
					(1 << PCD_DIN_PIN)	| 
					(1 << PCD_CS_PIN)	| 
					(1 << PCD_DC_PIN);
	pcd_enable();								// reset impulse

	SPCR		|=	(1 << SPE)	|				// SPE	- SPI Enable 
					(1 << MSTR) |				// MSTR - Master device 
					(1 << SPR0)	;				// SPR0 - Prescaler fclk/2 = 3.072MHz
	pcd_write(CMD,	PCD_FUNCTIONSET | 
					PCD_EXTENDEDINSTRUCTION);	// extended instruction set
	pcd_write(CMD,	PCD_SETBIAS|0x01);			// bias 1:48 - optimum bias value

												// for mux 1:48 optimum operation voltage is Ulcd = 6,06 * Uth
												// Ulcd = 3,06 + (Ucp6 to Ucp0) * 0,06
												// 6 < Ulcd < 8,05
												// command for operation voltage = 0x1 Ucp6 Ucp5 Ucp4 Ucp3 Ucp2 Ucp1 Ucp0
												// Ulcd = 0x11000010 = 7,02 V
	pcd_write(CMD,	PCD_SETVOP | 50); 
												// normal instruction set
	pcd_write(CMD,	PCD_FUNCTIONSET);			// horizontal addressing mode
	pcd_write(CMD,	PCD_DISPLAYCONTROL | 
					PCD_DISPLAYNORMAL);			// normal mode
}

// write command to display
// unsigned char - type (CMD or DATA)
// unsigned char - data
void pcd_write(unsigned char type, unsigned char data)
{
	if (type==CMD) 								// when command 
		CLR_BIT(PCD_PORT, PCD_DC_PIN);			// put data/command line active low
	else
		SET_BIT(PCD_PORT, PCD_DC_PIN);			// else high
		
	CLR_BIT(PCD_PORT, PCD_CS_PIN);				// chip enable - active low
	SPDR  = data;								// transmitting data
	while (!(SPSR & (1 << SPIF)));				// wait till data transmit
	SET_BIT(PCD_PORT, PCD_CS_PIN);				// chip disable - idle high
}

// clear display
void pcd_clearDisplay(void)
{
	memset(pcd_displayBuffer, 0x00, PCD_BUFFER_SIZE);	// null cache memory lcd
}

// update display
void pcd_updateDisplay(void)
{
	pcd_gotoxy(0, 0);							// set position x, y
	for (int i=0; i<PCD_BUFFER_SIZE; i++) 		// loop through cache memory lcd
		pcd_write(DATA, pcd_displayBuffer[i]);	// write data to lcd memory
}

// put character on display
// unsigned char - character
char pcd_putc(unsigned char character)
{
	if (	(character < PCD_FIRST_ASC) &&		// check if character is out of range
			(character > PCD_LAST_ASC))
		return 0;

	if ((pcd_BufferIndex % MAX_NUM_COLS) > (MAX_NUM_COLS - (PCD_CHAR_WIDTH-1) )) 
	{											// check if memory index not longer than 48 x 84
		pcd_BufferIndex = ((pcd_BufferIndex / MAX_NUM_COLS) + 1) * MAX_NUM_COLS;		
		if (pcd_BufferIndex  > PCD_BUFFER_SIZE)
			return 0;							// out of range
	}											// resize index on new row
	for (int i = 0; i < PCD_CHAR_WIDTH-1; i++) 	// loop through 5 bytes
		pcd_displayBuffer[pcd_BufferIndex++] = Characters[character - PCD_FIRST_ASC][i];		// read from ROM memory
	pcd_BufferIndex++;
	return 0;
}

// put string on display
// char* - string
void pcd_putstr(char *string)
{
	unsigned int i = 0;
	while (string[i] != '\0')					// loop through 5 bytes
		pcd_putc(string[i++]);					//read characters and increment index
}

// goto text position in line on pos
// unsigned char x - position 0 <= line <=  5
// unsigned char y - position 0 <= pos  <= 14
char pcd_gotoxy(unsigned char line, unsigned char pos)
{
	if ((line >= (	MAX_NUM_ROWS )) ||
		(pos  >=	MAX_NUM_COLS /PCD_CHAR_WIDTH  ))// check if x, y is in range
		return 0;										// out of range
	pcd_BufferIndex = pos * PCD_CHAR_WIDTH  + (line * MAX_NUM_COLS);	// calculate index memory
	return 1;
}

// update pixel position x, y
// unsigned char x - position 0 <= x <= 83
// unsigned char y - position 0 <= y <= 47
char pcd_updatePixelxy(unsigned char x, unsigned char y)
{ 
	if ((y >= (	MAX_NUM_ROWS * PCD_CHAR_HEIGHT)) ||
		(x >=	MAX_NUM_COLS   ))				// check if x, y is in range
		return 0;								// out of range
		
	pcd_write(CMD, PCD_FUNCTIONSET);			// normal instruction set, horizontal addressing mode
	pcd_write(CMD, (PCD_SETYADDR | (y / (PCD_CHAR_HEIGHT+1))));	// set x-position 
	pcd_write(CMD, (PCD_SETXADDR | x));			// set y-position
	pcd_BufferIndex = x + (y / PCD_CHAR_HEIGHT) * MAX_NUM_COLS;	// calculate index memory
	return 1;									// success return
}

// put pixel on display
// unsigned char x - position 0 <= x <= 83
// unsigned char y - position 0 <= y <= 47
char pcd_putPixel(unsigned char x, unsigned char y)
{ 
	if ((y >= (	MAX_NUM_ROWS * PCD_CHAR_HEIGHT)) ||
		(x >=	MAX_NUM_COLS   ))				// check if x, y is in range
		return 0;								// out of range
	
	pcd_BufferIndex = x + ((y / PCD_CHAR_HEIGHT) * MAX_NUM_COLS);	// calculate index memory
	pcd_displayBuffer[pcd_BufferIndex] |= 1 << (y % PCD_CHAR_HEIGHT);	// write pixel in byute of buffer
	return 1;									// success return
}

// put line by Bresenham algoritm
// https://en.wikipedia.org/wiki/Bresenham%27s_line_algorithm
//
// unsigned char x1 - start x position 0 <= x <= 83
// unsigned char y1 - start y position 0 <= y <= 47
// unsigned char x2 - end   x position 0 <= x <= 83
// unsigned char y2 - end   y position 0 <= y <= 47
char pcd_putLine(unsigned char x1, unsigned char y1, unsigned char x2, unsigned char y2)
{
	int16_t D;									// determinant
	int16_t delta_x		, delta_y;				// deltas
	int16_t trace_x = 1	, trace_y = 1;			// steps

	delta_x = x2 - x1;							// delta x
	delta_y = y2 - y1;							// delta y

	if (delta_x < 0)							// check if x2 > x1
	{	
		delta_x = -delta_x;						// negate delta x
		trace_x = -trace_x;						// negate step x
	}

	if (delta_y < 0) 							// check if y2 > y1
	{
		delta_y = -delta_y;						// negate delta y
		trace_y = -trace_y;						// negate step y
	}

	if (delta_y < delta_x) 						// Bresenham condition for m < 1 (dy < dx)
	{
		D = 2*delta_y - delta_x;					// calculate determinant
		pcd_putPixel(x1, y1);						// draw first pixel
		while (x1 != x2) 							// check if x1 equal x2
		{
			x1 += trace_x;							// update x1
			if (D >= 0) 							// check if determinant is positive
			{
				y1 += trace_y;							// update y1
				D -= 2*delta_x;							// update determinant
			}
			D += 2*delta_y;							// update determinant
			pcd_putPixel(x1, y1);					// draw next pixel
		}
	} else {									// if m > 1 (dy > dx) 
		D = delta_y - 2*delta_x;					// calculate determinant
		pcd_putPixel(x1, y1);						// draw first pixel
		while (y1 != y2) 							// check if y2 equal y1
		{
			y1 += trace_y;								// update y1
			if (D <= 0) 								// check if determinant is positive
			{
				x1 += trace_x;							// update y1
				D += 2*delta_y;							// update determinant
			}
			D -= 2*delta_x;							// update determinant
			pcd_putPixel(x1, y1);					// draw next pixel
		}
	}
	return 1;									// success return
}

